# mortal_write/views/characters.py

import streamlit as st
import streamlit.components.v1 as components
import os
import time
import json
import urllib.parse 
import random
import base64
import html
import re
import csv
import uuid
import threading
from datetime import datetime

# -----------------------------------------------------------------------------
# 🛡️ 导入防错处理
# -----------------------------------------------------------------------------
try:
    from database import save_avatar_file
except ImportError:
    # 如果找不到 database 模块，提供一个假的保存函数防止崩溃
    def save_avatar_file(file_obj, char_id):
        return None

try:
    from config import FEATURE_MODELS, DATA_DIR
except ImportError:
    DATA_DIR = "data"
    FEATURE_MODELS = {
        "character_extract": {"name": "角色提取", "default": "GPT_4o"},
        "books_arch_gen": {"name": "图谱生成", "default": "GPT_4o"}
    }

try:
    from logic import MODEL_MAPPING, OpenAI
except ImportError:
    MODEL_MAPPING = {}
    # 模拟 OpenAI 客户端结构，防止 AttributeError
    class MockChoices:
        def __init__(self, content):
            self.message = type('obj', (object,), {'content': content})

    class MockResponse:
        def __init__(self, content):
            self.choices = [MockChoices(content)]

    class MockCompletions:
        def create(self, *args, **kwargs):
            return MockResponse("[]") # 返回空JSON防止崩坏

    class MockChat:
        def __init__(self):
            self.completions = MockCompletions()

    class OpenAI:
        def __init__(self, api_key, base_url): 
            self.chat = MockChat()

# ==============================================================================
# 🛡️ 严格审计日志系统 (Characters 集成版)
# ==============================================================================

SYSTEM_LOG_PATH = os.path.join(DATA_DIR, "logs", "system_audit.csv")
_log_lock = threading.Lock()

def get_session_id():
    """获取或生成当前会话的唯一追踪ID"""
    if "session_trace_id" not in st.session_state:
        st.session_state.session_trace_id = str(uuid.uuid4())[:8]
    return st.session_state.session_trace_id

def log_audit_event(category, action, details, status="SUCCESS", module="CHARACTERS"):
    """
    执行严格的审计日志写入
    """
    try:
        os.makedirs(os.path.dirname(SYSTEM_LOG_PATH), exist_ok=True)
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        session_id = get_session_id()
        
        status_map = {"SUCCESS": "成功", "WARNING": "警告", "ERROR": "错误"}
        status_cn = status_map.get(status, status)
        
        module_map = {"CHARACTERS": "角色管理", "BOOKS": "书籍管理", "WRITER": "写作终端", "SETTINGS": "系统设置"}
        module_cn = module_map.get(module, module)

        if isinstance(details, (dict, list)):
            try: details = json.dumps(details, ensure_ascii=False)
            except: details = str(details)
            
        row = [timestamp, session_id, module_cn, category, action, status_cn, details]
        
        with _log_lock:
            file_exists = os.path.exists(SYSTEM_LOG_PATH)
            # 使用 utf-8-sig 以兼容 Windows Excel
            with open(SYSTEM_LOG_PATH, mode='a', newline='', encoding='utf-8-sig') as f:
                writer = csv.writer(f)
                if not file_exists or os.path.getsize(SYSTEM_LOG_PATH) == 0:
                    writer.writerow(['时间', '会话ID', '模块', '类别', '操作', '状态', '详情']) 
                writer.writerow(row)
    except Exception as e:
        print(f"❌ 审计日志写入失败: {e}")

# --- 0. 基础配置 ---
THEME_COLOR = "#2e7d32" 
THEME_LIGHT = "#e8f5e9"
RELATION_DIR = os.path.join(DATA_DIR, "relations") 
AVATAR_DIR = os.path.join(DATA_DIR, "avatars")

# --- 角色排序优先级 ---
ROLE_PRIORITY = {
    "主角": 0, "男主角": 0, 
    "女主角": 1, "双主角": 2, "妻子": 3, "夫君": 3, "暗恋者/伴侣": 3,
    "反派BOSS": 10, "大反派": 10,
    "主要配角": 20, "导师/师父": 21, "挚友/死党": 22, 
    "宿敌": 30, 
    "亲属(父母/兄妹)": 40, "金手指/系统化身": 41, "宠物/坐骑": 42,
    "次要配角": 50, 
    "小反派/炮灰": 60, 
    "路人": 99,
    "default": 99
}

if hasattr(st, "dialog"):
    dialog_decorator = st.dialog
else:
    dialog_decorator = st.experimental_dialog

# --- 🛠️ 本地工具函数 ---
def render_header(icon, title):
    """渲染页面标题"""
    st.markdown(f"## {icon} {title}")

# --- 1. 初始化 Session State ---
def init_option_state():
    defaults = {
        "role_options": ["主角", "女主角", "双主角", "主要配角", "次要配角", "反派BOSS", "小反派/炮灰", "导师/师父", "挚友/死党", "宿敌", "暗恋者/伴侣", "亲属(父母/兄妹)", "金手指/系统化身", "宠物/坐骑", "路人"],
        "gender_options": ["男", "女", "无性", "双性", "流体性别", "未知/神秘"],
        "race_options": ["人族", "精灵", "矮人", "兽人/半兽人", "龙族", "亡灵/丧尸", "魔族", "妖族", "仙/神族", "机械/仿生人/AI", "灵体/鬼魂", "异虫/怪兽", "吸血鬼", "狼人", "混血", "未知生物"]
    }
    for key, val in defaults.items():
        if key not in st.session_state: st.session_state[key] = val

def update_book_timestamp_by_book_id(book_id):
    if book_id: 
        try: st.session_state.db.update_book_timestamp(book_id)
        except: pass

# --------------------------------------------------------------------------
# 智能排序逻辑
# --------------------------------------------------------------------------
def get_role_priority(role_name):
    if not role_name: return 99
    r = role_name.strip()
    
    # 精确匹配优先
    if r in ROLE_PRIORITY:
        return ROLE_PRIORITY[r]
        
    # 模糊匹配
    if "男主" in r or r == "主角": return 0
    if any(k in r for k in ["女主", "双主角", "妻", "道侣", "伴侣", "红颜"]): return 1
    if any(k in r for k in ["反派", "BOSS", "魔尊", "始祖"]): return 2
    if any(k in r for k in ["主要配件", "导师", "师父", "挚友", "死党", "兄弟"]): return 3
    if any(k in r for k in ["宿敌", "亲属", "金手指", "系统"]): return 4
    if any(k in r for k in ["次要", "宠物", "坐骑"]): return 5
    if any(k in r for k in ["炮灰", "路人", "龙套"]): return 6
    
    return 99

# --- 智能总结函数 ---
def get_smart_short_label(text):
    """智能缩短关系标签"""
    if not text: return ""
    text = text.replace('（', '(').replace('）', ')').replace('、', '/')
    text = re.sub(r"\(.*?\)", "", text).strip() # 去括号
    if '/' in text: text = text.split('/')[0].strip() # 取首项
    if '&' in text: text = text.split('&')[0].strip()
    
    if '的' in text and len(text) > 4: # "救命恩人的女儿" -> "女儿"
        parts = text.split('的')
        candidate = parts[-1].strip()
        if candidate: text = candidate

    if len(text) > 4: return text[:3] + ".." # 硬截断
    return text

# --------------------------------------------------------------------------
#  核心交互：自定义选项模态弹窗
# --------------------------------------------------------------------------
@dialog_decorator("✨ 添加自定义选项")
def custom_option_dialog(list_key, widget_key):
    st.write("请输入新的选项名称：")
    new_val = st.text_input("输入内容", key=f"input_new_{list_key}")
    col_sub, col_can = st.columns([1, 1])
    
    if col_sub.button("✅ 确认并选中", type="primary", width="stretch"):
        if new_val and new_val.strip():
            if new_val not in st.session_state[list_key]:
                st.session_state[list_key].append(new_val)
            st.session_state[widget_key] = new_val
            
            # 🔥 审计：添加选项
            log_audit_event("基础配置", "添加自定义选项", {"类别": list_key, "值": new_val})
            
            st.rerun()
        else: st.warning("内容不能为空")

    if col_can.button("取消", width="stretch"):
        # 回退到默认第一个选项，防止死循环
        st.session_state[widget_key] = st.session_state[list_key][0]
        st.rerun()

def check_and_trigger_custom(selection, list_key, widget_key):
    if selection == "自定义...": custom_option_dialog(list_key, widget_key)

# --------------------------------------------------------------------------
# 核心交互：头像编辑模态弹窗
# --------------------------------------------------------------------------
@dialog_decorator("🖼️ 编辑角色头像")
def edit_avatar_dialog(char_id, current_avatar, char_name, current_book_id):
    st.caption(f"正在修改 **{char_name}** 的头像")
    
    col_prev, col_input = st.columns([1, 2.5], gap="medium", vertical_alignment="center")
    
    with col_prev:
        img_content = get_node_image_content(current_avatar)
        if img_content:
            st.image(img_content, width=110)
        else:
            st.info("无图")
            
    with col_input:
        new_file = st.file_uploader("上传新图片 (JPG/PNG)", type=['jpg', 'png'])
        new_url = st.text_input("输入图片 URL", value=current_avatar if isinstance(current_avatar, str) and current_avatar.startswith("http") else "")

    if st.button("💾 保存更改", type="primary", width="stretch"):
        final_path = current_avatar
        
        if new_file:
            saved_path = save_avatar_file(new_file, char_id)
            if saved_path:
                final_path = saved_path
        elif new_url and new_url != current_avatar:
            final_path = new_url
            
        if final_path != current_avatar:
            st.session_state.db.execute("UPDATE characters SET avatar=? WHERE id=?", (final_path, char_id))
            update_book_timestamp_by_book_id(current_book_id)
            
            # 🔥 审计：更新头像
            log_audit_event("角色管理", "更新头像", {"角色": char_name, "ID": char_id, "新路径": final_path})
            
            st.toast("✅ 头像已更新！")
            time.sleep(0.5)
            st.rerun()
        else:
            st.warning("未检测到更改")

# --------------------------------------------------------------------------
# 新增：字段编辑模态弹窗
# --------------------------------------------------------------------------
@dialog_decorator("✏️ 编辑字段")
def edit_field_dialog(char_id, field_name, current_value, char_name, current_book_id):
    st.caption(f"正在修改 **{char_name}** 的 {field_name}")
    
    # 统一使用文本输入框
    new_value = st.text_input(f"新的{field_name}", value=current_value or "")
    
    if st.button("💾 保存更改", type="primary", width="stretch"):
        if new_value != current_value:
            # 映射字段名到数据库列名
            field_mapping = {
                "姓名": "name",
                "定位": "role",
                "性别": "gender",
                "种族": "race",
                "出身背景": "origin",
                "金手指/核心能力": "cheat_ability",
                "当前境界/等级": "power_level",
                "职业/天赋": "profession"
            }
            
            db_field = field_mapping.get(field_name)
            if db_field:
                st.session_state.db.execute(f"UPDATE characters SET {db_field}=? WHERE id=?", (new_value, char_id))
                update_book_timestamp_by_book_id(current_book_id)
                
                # 🔥 审计：更新字段
                log_audit_event("角色管理", f"更新{field_name}", {"角色": char_name, "ID": char_id, "新值": new_value})
                
                st.toast(f"✅ {field_name} 已更新！")
                time.sleep(0.5)
                st.rerun()
            else:
                st.warning("无法识别字段名")

# --------------------------------------------------------------------------
# 🔥 修复版：图片处理 & URL 生成 (增强版头像获取)
# --------------------------------------------------------------------------

def get_node_image_content(path_or_url):
    """
    🔥 增强版头像获取函数
    支持多种路径格式：
    1. HTTP/HTTPS URL
    2. Data URI (base64)
    3. 绝对路径
    4. 相对路径 (相对于 DATA_DIR/avatars/)
    5. 数据库存储的相对路径
    """
    if not path_or_url:
        return None
    
    path_or_url = str(path_or_url).strip()
    
    # 1. 如果是 HTTP URL 或 Data URI，直接返回
    if path_or_url.startswith(("http://", "https://", "data:")):
        return path_or_url
    
    # 2. 尝试直接路径 (处理 Windows 反斜杠)
    path_or_url = os.path.normpath(path_or_url)
    
    if os.path.exists(path_or_url):
        try:
            return local_file_to_data_uri(path_or_url)
        except:
            pass
    
    # 3. 尝试在 avatar 目录下查找
    avatar_filename = os.path.basename(path_or_url)
    avatar_path = os.path.join(AVATAR_DIR, avatar_filename)
    if os.path.exists(avatar_path):
        try:
            return local_file_to_data_uri(avatar_path)
        except:
            pass
    
    # 4. 尝试原始路径（可能数据库存储的是相对路径）
    if not os.path.isabs(path_or_url):
        # 尝试在 data 目录下
        data_path = os.path.join(DATA_DIR, path_or_url)
        if os.path.exists(data_path):
            try:
                return local_file_to_data_uri(data_path)
            except:
                pass
        
        # 尝试在项目根目录下
        root_path = os.path.join(os.getcwd(), path_or_url)
        if os.path.exists(root_path):
            try:
                return local_file_to_data_uri(root_path)
            except:
                pass
    
    # 5. 如果所有尝试都失败，返回 None
    return None

def local_file_to_data_uri(file_path):
    """将本地文件转换为 data URI"""
    try:
        with open(file_path, "rb") as img_file:
            b64_string = base64.b64encode(img_file.read()).decode('utf-8')
            ext = file_path.split('.')[-1].lower() if '.' in file_path else 'jpg'
            if ext == "png": mime_type = "image/png"
            elif ext in ["jpg", "jpeg"]: mime_type = "image/jpeg"
            elif ext == "gif": mime_type = "image/gif"
            else: mime_type = "image/jpeg"
            
            return f"data:{mime_type};base64,{b64_string}"
    except Exception as e:
        print(f"转换图片失败 {file_path}: {e}")
        return None

def generate_bing_search_image(keyword):
    """生成 Bing 搜索图片 URL"""
    if not keyword: 
        return ""
    encoded = urllib.parse.quote(keyword)
    # 使用 Bing 缩略图服务，比较稳定
    return f"https://tse2.mm.bing.net/th?q={encoded}&w=300&h=300&c=7&rs=1&p=0"

def get_default_avatar(name):
    """获取默认头像（当没有头像时）"""
    if not name:
        return "https://api.dicebear.com/9.x/adventurer/svg?seed=Unknown&flip=true"
    return f"https://api.dicebear.com/9.x/adventurer/svg?seed={urllib.parse.quote(name)}&flip=true"

def save_relations_to_disk(book_id, relations_data):
    if not os.path.exists(RELATION_DIR): 
        os.makedirs(RELATION_DIR, exist_ok=True)
    file_path = os.path.join(RELATION_DIR, f"book_{book_id}.json")
    try:
        with open(file_path, "w", encoding="utf-8") as f:
            json.dump(relations_data, f, ensure_ascii=False, indent=2)
        return True
    except Exception as e:
        print(f"Save Error: {e}")
        return False

def load_relations_from_disk(book_id):
    file_path = os.path.join(RELATION_DIR, f"book_{book_id}.json")
    if os.path.exists(file_path):
        try:
            with open(file_path, "r", encoding="utf-8") as f: 
                return json.load(f)
        except Exception as e:
            print(f"Load Error: {e}")
            return []
    return []

# ==============================================================================
# 🔥 新增：自定义模型解析器 (与 Books 模块保持一致)
# ==============================================================================
def _resolve_ai_client(engine, assigned_key):
    """
    智能解析客户端：支持原生模型 + 自定义模型(CUSTOM::)
    """
    # 1. 拦截自定义模型
    if assigned_key and str(assigned_key).startswith("CUSTOM::"):
        try:
            # 提取真实名称
            target_name = assigned_key.split("::", 1)[1]
            
            # 从数据库读取配置
            settings = engine.get_config_db("ai_settings", {})
            custom_list = settings.get("custom_model_list", [])
            
            # 查找匹配项
            for m in custom_list:
                if m.get("name") == target_name:
                    api_key = m.get("key")
                    base_url = m.get("base")
                    model_id = m.get("api_model")
                    
                    if not api_key or not base_url:
                        return None, None, None
                        
                    # 实例化 OpenAI (使用 logic 中导入的类)
                    client = OpenAI(api_key=api_key, base_url=base_url)
                    return client, model_id, "custom"
            
            print(f"❌ 未找到自定义模型配置: {target_name}")
            return None, None, None
        except Exception as e:
            print(f"❌ 自定义模型解析失败: {e}")
            return None, None, None

    # 2. 原生模型走默认逻辑
    return engine.get_client(assigned_key)

# --------------------------------------------------------------------------
# 数据库 & AI 提取逻辑
# --------------------------------------------------------------------------
def ensure_schema_compatibility(db_mgr):
    new_cols = [
        "race", "desc", "avatar", "is_major",
        "origin", "profession", "cheat_ability", "power_level", 
        "ability_limitations", "appearance_features", "signature_sign",
        "relationship_to_protagonist", "social_role", "debts_and_feuds"
    ]
    for col in new_cols:
        try: 
            # 简单检查列是否存在，不存在则添加
            db_mgr.query(f"SELECT {col} FROM characters LIMIT 1")
        except: 
            try: 
                db_mgr.execute(f"ALTER TABLE characters ADD COLUMN {col} TEXT")
            except: 
                pass

def repair_json_content(content):
    """
    🔥 强力修复 JSON 字符串
    """
    if not content: 
        return "[]"
    
    # 1. 移除 Markdown 代码块标记 (包括 json, text 等标签)
    content = re.sub(r'^```[a-zA-Z]*\s*', '', content, flags=re.MULTILINE)
    content = re.sub(r'^```\s*', '', content, flags=re.MULTILINE)
    content = re.sub(r'\s*```$', '', content, flags=re.MULTILINE)
    
    # 2. 尝试提取最外层的列表 [ ... ]
    match = re.search(r'\[.*\]', content, re.DOTALL)
    if match:
        content = match.group(0)
    
    # 3. 移除尾随逗号 (Trailing Commas)
    content = re.sub(r',(\s*\})', r'\1', content)
    content = re.sub(r',(\s*\])', r'\1', content)
    
    return content.strip()

def ai_extract_characters(engine, db_mgr, current_book, current_book_id, progress_callback=None):
    
    # 🔥 审计：启动提取
    log_audit_event("AI辅助", "启动角色提取", {"书籍": current_book['title']})
    
    feature_key = "character_extract"
    assigned_model_key = engine.get_config_db("model_assignments", {}).get(feature_key, FEATURE_MODELS[feature_key]['default'])
    
    # 🔥 修复：使用支持自定义模型的解析器
    client, model_name, _ = _resolve_ai_client(engine, assigned_model_key)
    
    feat_name = FEATURE_MODELS[feature_key]['name']
    display_model_name = MODEL_MAPPING.get(assigned_model_key, {}).get('name', model_name)
    
    if not client: 
        log_audit_event("AI辅助", "角色提取失败", {"原因": "未配置模型"}, status="ERROR")
        return False, f"❌ AI 模型未配置。请在【系统设置】->【功能调度】中为 [{feat_name}] 选择模型并保存。"

    if progress_callback: 
        progress_callback(10, "STEP 1/6: 正在扫描现有角色库...")
    existing_res = db_mgr.query("SELECT name FROM characters WHERE book_id=?", (current_book_id,))
    existing_names = {r['name'] for r in existing_res} if existing_res else set()

    if progress_callback: 
        progress_callback(20, "STEP 2/6: 正在读取小说内容...")
    content_snippet = ""
    if hasattr(engine, 'get_book_content_prefix'):
        content_snippet = engine.get_book_content_prefix(current_book_id, length=15000)
    
    if progress_callback: 
        progress_callback(40, f"STEP 3/6: 正在调用 {display_model_name} 深度分析 (这可能需要十几秒)...")
    
    prompt = f"""
    请深入分析小说《{current_book['title']}》。
    {f"参考小说前文片段：{content_snippet[:3500]}..." if content_snippet else "请基于你的知识库。"}
    
    任务：提取该小说中最重要的 5-10 个角色，并补充详细的网文设定属性。
    
    要求：
    1. 必须返回纯粹的 JSON 数组格式。
    2. 不要包含 markdown 标记。
    3. 严禁使用尾随逗号（trailing commas）。
    4. 排除已存在的角色：{list(existing_names)}
    
    返回 JSON 结构示例：
    [
        {{
            "name": "角色名", "gender": "男/女", "race": "种族", "role": "主角/反派...",
            "desc": "基础外貌和性格描述",
            "is_major": true,
            "origin": "家族弃子",
            "profession": "炼药师",
            "cheat_ability": "骨灵冷火",
            "power_level": "斗之气三段",
            "relationships": [
                 {{"target": "另一个角色名", "label": "义父"}},
                 {{"target": "另一个角色名", "label": "宿敌"}}
            ]
        }}
    ]
    """
    
    try:
        response = client.chat.completions.create(
            model=model_name,
            messages=[{"role": "user", "content": prompt}],
            temperature=0.3
        )
        
        if progress_callback: 
            progress_callback(70, "STEP 4/6: 解析数据结构与关系...")
        
        raw_content = response.choices[0].message.content.strip()
        cleaned_json = repair_json_content(raw_content)

        try:
            char_list = json.loads(cleaned_json)
        except json.JSONDecodeError as e:
            print(f"JSON Parse Error: {e} | Content: {raw_content}")
            log_audit_event("AI辅助", "角色提取失败", {"原因": "JSON解析错误", "详情": str(e)}, status="ERROR")
            return False, f"AI 返回格式有误，自动修复失败。请重试。(Error: {e})"

        final_list = []
        total_items = len(char_list)
        
        for i, c in enumerate(char_list):
            if not c.get('name'): 
                continue
            
            # 动态更新进度
            current_progress = 70 + int((i / total_items) * 25)
            if progress_callback: 
                progress_callback(current_progress, f"STEP 5/6: 准备 {c['name']} 的数据...")
            
            avatar_url = c.get('avatar', '').strip()
            if not avatar_url or not avatar_url.startswith("http"):
                search_query = f"{current_book['title']} {c['name']} 插画"
                avatar_url = generate_bing_search_image(search_query)
            
            c['avatar'] = avatar_url
            final_list.append(c)
        
        if progress_callback: 
            progress_callback(100, "✅ 分析完成！")
            
        # 🔥 审计：提取成功
        log_audit_event("AI辅助", "角色提取完成", {"发现数量": len(final_list)})
        return True, final_list
        
    except Exception as e:
        log_audit_event("AI辅助", "角色提取失败", {"错误": str(e)}, status="ERROR")
        return False, f"提取失败: {str(e)}"

# --------------------------------------------------------------------------
# 🔥 修复版：核心渲染逻辑 (CDN 链接修正)
# --------------------------------------------------------------------------

def generate_graph_html(nodes, edges, height="450px"):
    
    nodes_json = json.dumps(nodes)
    edges_json = json.dumps(edges)
    
    # 🔥 核心修复点：使用更稳定的 cdnjs
    vis_js_url = "https://cdnjs.cloudflare.com/ajax/libs/vis/4.21.0/vis.min.js"
    vis_css_url = "https://cdnjs.cloudflare.com/ajax/libs/vis/4.21.0/vis.min.css"

    html_template = f"""
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="utf-8">
        <script type="text/javascript" src="{vis_js_url}"></script>
        <link href="{vis_css_url}" rel="stylesheet" type="text/css" />
        <style type="text/css">
            #mynetwork {{
                width: 100%;
                height: {height};
                border: 1px solid #eee;
                background-color: #ffffff;
            }}
            
            /* 🔥 恢复操作栏，确保居中控件显示 */
            div.vis-network div.vis-manipulation {{
                background-color: #f8f9fa;
                border-bottom: 1px solid #e9ecef;
                padding: 8px;
                display: flex !important;
                justify-content: center;
                align-items: center;
            }}
            
            /* 美化操作按钮 */
            div.vis-manipulation button {{
                background-color: #fff;
                border: 1px solid #dee2e6;
                border-radius: 4px;
                padding: 6px 12px;
                margin: 0 4px;
                cursor: pointer;
                font-size: 12px;
                color: #495057;
                transition: all 0.2s;
            }}
            
            div.vis-manipulation button:hover {{
                background-color: #e9ecef;
                border-color: #adb5bd;
            }}
            
            /* 🔥 优化：Tooltip 样式 - 确保显示完整，自动换行 */
            div.vis-tooltip {{
                position: absolute;
                background-color: #ffffff;
                border: 1px solid #e0e0e0;
                border-radius: 8px;
                box-shadow: 0 4px 15px rgba(0,0,0,0.1);
                padding: 12px;
                font-family: "Microsoft YaHei", sans-serif;
                font-size: 12px;
                line-height: 1.5;
                color: #333;
                width: auto;
                max-width: 380px;
                max-height: 450px;
                overflow-y: auto;
                z-index: 9999;
                /* 🔥 关键修复：确保长文本自动换行 */
                word-wrap: break-word;
                white-space: normal;
                overflow-wrap: break-word;
            }}
            div.vis-tooltip strong {{ 
                color: #2e7d32; 
                font-size: 14px; 
                display: block; 
                margin-bottom: 6px; 
                font-weight: bold;
                word-break: break-word;
            }}
            div.vis-tooltip span.meta {{ 
                color: #888; 
                font-size: 11px; 
                display: block; 
                margin-bottom: 8px; 
                border-bottom: 1px dashed #eee; 
                padding-bottom: 6px;
                word-break: break-word;
            }}
            div.vis-tooltip div.info-line {{
                font-size: 11px;
                color: #666;
                margin: 2px 0;
                word-break: break-word;
            }}
            div.vis-tooltip div.desc {{
                font-size: 11px;
                color: #666;
                margin-top: 8px;
                line-height: 1.4;
                word-break: break-word;
            }}
            
            /* 🔥 修复：边工具提示样式 - 简化为简单文本 */
            .edge-tooltip {{
                position: absolute !important;
                background-color: #ffffff !important;
                border: 1px solid #e0e0e0 !important;
                border-radius: 8px !important;
                box-shadow: 0 4px 15px rgba(0,0,0,0.1) !important;
                padding: 12px !important;
                font-family: "Microsoft YaHei", sans-serif !important;
                font-size: 12px !important;
                line-height: 1.5 !important;
                color: #333 !important;
                width: auto !important;
                max-width: 400px !important;
                z-index: 9999 !important;
                word-wrap: break-word !important;
                white-space: normal !important;
                overflow-wrap: break-word !important;
            }}
        </style>
    </head>
    <body>
    <div id="mynetwork"></div>
    <script type="text/javascript">
        // 1. 加载数据
        var nodes = new vis.DataSet({nodes_json});
        var edges = new vis.DataSet({edges_json});
        var container = document.getElementById('mynetwork');
        var data = {{ nodes: nodes, edges: edges }};
        
        // 2. 配置项 - 🔥 恢复操作栏
        var options = {{
            locale: 'cn',
            nodes: {{
                shape: 'dot', 
                size: 32,
                font: {{ size: 14, color: '#333333', strokeWidth: 3, strokeColor: '#ffffff' }},
                borderWidth: 3, 
                shadow: true
            }},
            edges: {{
                width: 3,  // 🔥 增加线宽，让连线更明显
                color: {{ color: '#444444', highlight: '#2e7d32' }},  // 🔥 加深连线颜色
                smooth: {{ type: 'continuous', roundness: 0.5 }}, 
                arrows: 'to',
                font: {{
                    size: 11,
                    color: '#333333',
                    align: 'middle',
                    background: 'rgba(255,255,255,0.85)', 
                    strokeWidth: 0
                }}
            }},
            physics: {{
                enabled: true,
                forceAtlas2Based: {{ 
                    gravitationalConstant: -200,  /* 🔥 增加排斥力，让节点更分散 */
                    springLength: 250,  /* 🔥 增加弹簧长度，让连线更长 */
                    damping: 0.4
                }},
                solver: 'forceAtlas2Based'
            }},
            interaction: {{ 
                hover: true, 
                zoomView: true, 
                dragView: true,
                navigationButtons: true,
                keyboard: true
            }},
            manipulation: {{
                enabled: false,
                initiallyActive: false
            }}
        }};
        
        // 3. 初始化网络
        try {{
            var network = new vis.Network(container, data, options);
            
            // 4. 稳定后适配视图
            network.once("stabilizationIterationsDone", function() {{
                network.fit({{ animation: {{ duration: 1000 }} }});
            }});
            
            // 🔥 修复：确保边的悬停提示能够显示
            network.on("hoverEdge", function(params) {{
                var edgeId = params.edge;
                if (edgeId) {{
                    var edgeData = edges.get(edgeId);
                    if (edgeData && edgeData.title) {{
                        // 创建自定义工具提示
                        var tooltip = document.createElement('div');
                        tooltip.className = 'edge-tooltip';
                        tooltip.innerHTML = edgeData.title;
                        tooltip.style.position = 'absolute';
                        tooltip.style.left = params.event.center.x + 'px';
                        tooltip.style.top = params.event.center.y + 'px';
                        tooltip.style.zIndex = '10000';
                        
                        // 移除现有的工具提示
                        var existingTooltips = document.getElementsByClassName('edge-tooltip');
                        while(existingTooltips.length > 0) {{
                            existingTooltips[0].parentNode.removeChild(existingTooltips[0]);
                        }}
                        
                        document.body.appendChild(tooltip);
                    }}
                }}
            }});
            
            network.on("blurEdge", function(params) {{
                // 移除所有边工具提示
                var tooltips = document.getElementsByClassName('edge-tooltip');
                while(tooltips.length > 0) {{
                    tooltips[0].parentNode.removeChild(tooltips[0]);
                }}
            }});
            
        }} catch (err) {{
            console.error("Vis.js Init Error: ", err);
            document.getElementById('mynetwork').innerHTML = '<div style="padding:20px;color:red">渲染引擎加载失败，请检查网络连接 (CDN)</div>';
        }}
    </script>
    </body>
    </html>
    """
    return html_template

# --------------------------------------------------------------------------
# 主渲染函数
# --------------------------------------------------------------------------

def ai_generate_relation_map(engine, book_id, char_list, client, model_name):
    """
    🔥 本地化的人物关系生成函数，替代 engine 中可能出错的版本。
    使用 repair_json_content 确保 JSON 格式稳健。
    """
    if not char_list:
        return False, "没有足够角色数据"

    names = [c['name'] for c in char_list]
    name_map = {c['name']: c['id'] for c in char_list}
    
    snippet = ""
    if hasattr(engine, 'get_book_content_prefix'):
        snippet = engine.get_book_content_prefix(book_id, length=10000)

    # 🔥 修复：明确要求纯JSON，防止 AI 废话
    # 🔥 优化：增加"严格使用提供人名"的要求，防止AI创造别名导致断连
    # 🔥 修改：要求AI提供关系起因和提炼后的关系描述
    prompt = f"""
    基于以下小说片段和角色列表，生成人物关系图谱数据。
    
    【角色列表 (严格遵守此名单)】
    {json.dumps(names, ensure_ascii=False)}

    【小说片段】
    {snippet[:4000]}...

    【任务要求】
    1. 仅关注【角色列表】中已有的角色。
    2. 返回 JSON 数组。
    3. 格式：[{{ "source": "角色A名字", "target": "角色B名字", "label": "提炼后的关系描述(如:师徒, 仇敌)", "reason": "因为什么事情产生了这段关系", "weight": 1-5 }}]
    4. "weight" 代表关系亲密度/重要性 (1=普通, 5=极其重要)。
    5. "reason" 字段：简要描述因为什么事情产生了这段关系，例如："在青云门学艺时拜师"、"因争夺宝物结仇"。
    6. 不要包含 Markdown 格式，不要代码块标记。
    7. 重要：source 和 target 必须完全严格匹配【角色列表】中的名字，禁止使用文中昵称或别名，否则关系将丢失。
    """
    
    try:
        response = client.chat.completions.create(
            model=model_name,
            messages=[{"role": "user", "content": prompt}],
            temperature=0.3
        )
        
        raw_content = response.choices[0].message.content.strip()
        cleaned_json = repair_json_content(raw_content)
        
        data = json.loads(cleaned_json)
        
        final_rels = []
        for r in data:
            s_name = r.get('source')
            t_name = r.get('target')
            
            # 尝试匹配 ID
            s_id = name_map.get(s_name)
            t_id = name_map.get(t_name)
            
            # 简单模糊匹配回退 (仅当精确匹配失败时)
            if not s_id:
                for k,v in name_map.items():
                    if k in s_name or s_name in k: s_id = v; break
            if not t_id:
                for k,v in name_map.items():
                    if k in t_name or t_name in k: t_id = v; break
            
            if s_id and t_id and s_id != t_id:
                final_rels.append({
                    "source": s_id,
                    "target": t_id,
                    "label": r.get('label', '未知'),
                    "reason": r.get('reason', ''),
                    "weight": r.get('weight', 1)
                })
        
        return True, final_rels
        
    except Exception as e:
        return False, f"生成异常: {str(e)}"

def render_characters(engine, current_book_arg=None):
    """Render character profile page"""
    db_mgr = st.session_state.db
    ensure_schema_compatibility(db_mgr)
    init_option_state() 
    
    render_header("👥", "角色档案")
    
    # CSS 样式优化 - 修复所有问题
    st.markdown(f"""
    <style>
    input[type="checkbox"] {{ accent-color: {THEME_COLOR} !important; }}
    
    [data-testid="StyledFullScreenButton"] {{ display: none !important; }}
    button[kind="header"] {{ display: none !important; }}

    [data-testid='stFileUploaderDropzone'] {{ 
        border: 1px dashed #bbb !important; 
        background-color: #fafafa !important; 
        border-radius: 4px !important; 
        padding: 0 !important; 
        min-height: 40px !important; 
        height: 40px !important;
        display: flex !important; 
        align-items: center !important; 
        justify-content: center !important;
        cursor: pointer;
    }}
    [data-testid='stFileUploaderDropzone']:hover {{
        border-color: {THEME_COLOR} !important;
        background-color: #f1f8e9 !important;
    }}
    [data-testid='stFileUploaderDropzone'] > div > div > svg, 
    [data-testid='stFileUploaderDropzone'] > div > div > small, 
    [data-testid='stFileUploaderDropzone'] span {{ display: none !important; }}
    
    [data-testid='stFileUploaderDropzone']::after {{ 
        content: "📷 点击更换"; 
        display: block !important;
        color: #666; font-size: 13px; font-weight: 500; visibility: visible !important;
    }}
    [data-testid='stFileUploader'] button[kind="secondary"] {{ display: none !important; }}
    
    /* 🔥 确保列容器对齐一致 */
    div[data-testid="column"] {{
        display: flex !important;
        flex-direction: column !important;
        justify-content: center !important;
    }}
    
    /* 头像图片样式 - 确保绝对居中 */
    .char-avatar-img {{
        width: 80px !important;
        height: 80px !important;
        object-fit: cover !important;
        border: 2px solid #ddd !important;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1) !important;
        margin: 0 auto !important;
        display: block !important;
        margin-bottom: 8px !important;
        cursor: pointer !important;
        transition: all 0.3s ease !important;
    }}
    
    .char-avatar-img:hover {{
        border-color: {THEME_COLOR} !important;
        transform: scale(1.05) !important;
        box-shadow: 0 4px 12px rgba(46, 125, 50, 0.2) !important;
    }}
    
    /* 🔥 移除Expander内部的默认边距干扰 */
    .streamlit-expanderHeader {{
        padding: 8px 12px !important;
    }}
    
    .streamlit-expanderContent {{
        padding: 12px 12px 8px 12px !important;
    }}
    
    /* 全局输入框微调 - 修复文字显示 */
    .stTextInput input, .stTextArea textarea {{
        padding-top: 8px !important; 
        padding-bottom: 8px !important;
    }}
    
    /* 列表卡片样式：去除 Expander 的默认边框感，改用清爽底色 */
    [data-testid="stExpander"] {{
        background-color: #ffffff !important;
        border: 1px solid #e0e0e0 !important;
        border-radius: 8px !important;
        box-shadow: 0 1px 2px rgba(0,0,0,0.05) !important;
        margin-bottom: 8px !important;
    }}
    
    /* 解决列对齐问题：强制垂直居中 */
    .stColumn {{
        display: flex !important;
        flex-direction: column !important;
        justify-content: center !important;
    }}

    /* 身份背景 Tag 样式 */
    .char-tag {{
        display: inline-block;
        background: #f1f8e9; color: #33691e;
        padding: 2px 8px; border-radius: 12px;
        font-size: 11px; margin-right: 4px; border: 1px solid #dcedc8;
    }}
    
    /* 关系 Badge 样式 */
    .rel-badge {{
        display: inline-block;
        background: #e3f2fd; color: #0d47a1;
        padding: 2px 8px; border-radius: 4px;
        font-size: 10px; margin-right: 5px; margin-bottom: 3px;
        border: 1px solid #bbdefb;
    }}
    
    /* 🔥 移除不必要的横线 */
    .stExpander hr {{
        display: none !important;
    }}
    
    /* 紧凑布局：减少内边距 */
    .stExpander > div {{
        padding-top: 6px !important;
        padding-bottom: 6px !important;
    }}
    
    /* 🔥 字段标签样式 */
    .field-label {{
        font-size: 11px !important;
        font-weight: 600 !important;
        color: #555 !important;
        margin-bottom: 3px !important;
        display: block !important;
    }}
    
    .field-hint {{
        font-size: 10px !important;
        color: #888 !important;
        margin-top: 1px !important;
        margin-bottom: 8px !important;
        font-style: italic !important;
    }}
    
    /* 🔥 移除Expander标题行的默认图标 */
    .streamlit-expanderHeader svg {{
        display: none !important;
    }}
    
    /* 🔥 修复头像形状切换 */
    .avatar-shape-toggle {{
        margin-bottom: 8px !important;
    }}
    
    /* 🔥 确保按钮对齐 */
    .char-action-buttons {{
        margin-top: 12px !important;
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
    }}
    
    /* 🔥 修复底部按钮对齐 */
    .action-buttons-row {{
        display: flex !important;
        flex-direction: row !important;
        align-items: center !important;
        gap: 8px !important;
        width: 100% !important;
    }}
    
    /* 🔥 确保文本区域不遮挡 */
    .stTextArea textarea {{
        min-height: 100px !important;
    }}
    
    /* 🔥 修复所有输入框的垂直居中 */
    .stTextInput > div > div {{
        display: flex !important;
        align-items: center !important;
        min-height: 38px !important;
    }}
    
    /* 🔥 确保文本框文字垂直居中 */
    .stTextInput input {{
        padding-top: 8px !important;
        padding-bottom: 8px !important;
    }}
    
    /* 🔥 修复选择框的箭头图标位置 */
    div[data-baseweb="select"] > div > div > svg {{
        position: absolute !important;
        right: 10px !important;
        top: 50% !important;
        transform: translateY(-50%) !important;
        margin-top: 0 !important;
    }}
    
    /* 🔥 修复书籍选择下拉框样式 */
    div[data-testid="stSelectbox"] > div > div > div {{
        padding-right: 30px !important; /* 为箭头留出空间 */
    }}
    
    /* 🔥 字段显示行样式 */
    .field-display-row {{
        display: flex !important;
        align-items: center !important;
        justify-content: space-between !important;
        padding: 8px 0 !important;
        border-bottom: 1px dashed #eee !important;
        cursor: pointer !important;
        transition: all 0.2s ease !important;
    }}
    
    .field-display-row:hover {{
        background-color: #f9f9f9 !important;
        padding-left: 5px !important;
        padding-right: 5px !important;
        border-radius: 4px !important;
    }}
    
    .field-display-label {{
        font-weight: 600 !important;
        color: #666 !important;
        font-size: 12px !important;
        flex: 0 0 100px !important;
    }}
    
    .field-display-value {{
        color: #333 !important;
        font-size: 13px !important;
        flex-grow: 1 !important;
        margin-left: 10px !important;
        padding: 4px 8px !important;
        border-radius: 4px !important;
        transition: all 0.2s ease !important;
    }}
    
    .field-display-value:hover {{
        background-color: {THEME_LIGHT} !important;
        color: {THEME_COLOR} !important;
    }}
    
    /* 🔥 修复头像形状切换按钮样式 - 无边框 */
    .stRadio > div {{
        display: flex !important;
        justify-content: center !important;
        gap: 15px !important;
        background: none !important;
        border: none !important;
        padding: 0 !important;
    }}
    
    .stRadio > div > label {{
        margin-bottom: 0 !important;
        padding: 6px 12px !important;
        border-radius: 6px !important;
        border: none !important;
        background: none !important;
        cursor: pointer !important;
        transition: all 0.2s ease !important;
        font-size: 14px !important;
    }}
    
    .stRadio > div > label:hover {{
        background: #f5f5f5 !important;
    }}
    
    .stRadio > div > label[data-checked="true"] {{
        background: {THEME_LIGHT} !important;
        color: {THEME_COLOR} !important;
        font-weight: bold !important;
    }}
    
    /* 🔥 修复标题中的编辑按钮 */
    .title-edit-btn {{
        background: transparent !important;
        border: 1px solid #ddd !important;
        color: #666 !important;
        padding: 2px 8px !important;
        font-size: 11px !important;
        border-radius: 3px !important;
        cursor: pointer !important;
        transition: all 0.2s ease !important;
    }}
    
    .title-edit-btn:hover {{
        border-color: {THEME_COLOR} !important;
        color: {THEME_COLOR} !important;
        background: {THEME_LIGHT} !important;
    }}
    
    /* 🔥 修复下拉框文字居中 */
    div[data-baseweb="select"] > div {{
        padding-top: 8px !important;
        padding-bottom: 8px !important;
        min-height: 38px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: flex-start !important;
    }}

    /* 🔥 确保下拉框内部文字垂直和水平居中 */
    .stSelectbox > div > div > div {{
        padding-top: 8px !important;
        padding-bottom: 8px !important;
        min-height: 38px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: flex-start !important;
    }}
    
    /* 🔥 修复下拉菜单项文字居中 */
    div[role="listbox"] div {{
        padding-top: 6px !important;
        padding-bottom: 6px !important;
        line-height: 1.4 !important;
        min-height: 36px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: flex-start !important;
    }}
    
    /* 🔥 角色标题样式 */
    .char-title {{
        display: flex !important;
        align-items: center !important;
        justify-content: space-between !important;
        width: 100% !important;
    }}
    
    .char-title-text {{
        flex-grow: 1 !important;
        font-weight: 600 !important;
        color: #333 !important;
        font-size: 14px !important;
    }}
    
    /* 🔥 新增：简化形状选择器样式 */
    .simple-shape-toggle {{
        display: flex !important;
        gap: 10px !important;
        margin-bottom: 10px !important;
    }}
    
    .simple-shape-btn {{
        padding: 6px 12px !important;
        border: 1px solid #ddd !important;
        border-radius: 6px !important;
        background: white !important;
        cursor: pointer !important;
        font-size: 16px !important;
        transition: all 0.2s ease !important;
    }}
    
    .simple-shape-btn:hover {{
        background: #f5f5f5 !important;
    }}
    
    .simple-shape-btn.active {{
        background: {THEME_LIGHT} !important;
        color: {THEME_COLOR} !important;
        border-color: {THEME_COLOR} !important;
        font-weight: bold !important;
    }}
    
    /* 🔥 新增：两列布局样式 - 简化版 */
    .two-column-simple {{
        display: flex !important;
        flex-direction: row !important;
        gap: 30px !important;
        margin-bottom: 16px !important;
    }}
    
    .left-column {{
        flex: 0 0 40% !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 15px !important;
    }}
    
    .right-column {{
        flex: 0 0 40% !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 15px !important;
    }}
    
    .avatar-section {{
        text-align: center !important;
        margin-bottom: 15px !important;
    }}
    
    .avatar-img {{
        width: 100px !important;
        height: 100px !important;
        object-fit: cover !important;
        border-radius: 8px !important;
        border: 2px solid #dee2e6 !important;
        cursor: pointer !important;
        transition: all 0.3s ease !important;
        margin-bottom: 10px !important;
    }}
    
    .avatar-img:hover {{
        border-color: {THEME_COLOR} !important;
        transform: scale(1.05) !important;
    }}
    
    .field-row {{
        display: flex !important;
        flex-direction: column !important;
        margin-bottom: 12px !important;
    }}
    
    .field-label {{
        font-size: 11px !important;
        font-weight: 600 !important;
        color: #666 !important;
        margin-bottom: 4px !important;
    }}
    
    .field-value {{
        font-size: 13px !important;
        color: #333 !important;
        padding: 8px 12px !important;
        border-radius: 4px !important;
        background: #f8f9fa !important;
        border: 1px solid #e9ecef !important;
        min-height: 38px !important;
        display: flex !important;
        align-items: center !important;
        cursor: pointer !important;
        transition: all 0.2s ease !important;
        word-break: break-word !important;
    }}
    
    .field-value:hover {{
        background: #e9ecef !important;
        border-color: #ced4da !important;
    }}
    
    /* 🔥 新增：可点击头像容器 */
    .clickable-avatar-container {{
        position: relative;
        width: 100px;
        height: 100px;
        margin: 0 auto 10px auto;
    }}
    
    .clickable-avatar-img {{
        width: 100% !important;
        height: 100% !important;
        object-fit: cover !important;
        border-radius: 8px !important;
        border: 2px solid #dee2e6 !important;
        cursor: pointer !important;
        transition: all 0.3s ease !important;
    }}
    
    .clickable-avatar-img:hover {{
        border-color: {THEME_COLOR} !important;
        transform: scale(1.05) !important;
    }}
    
    .avatar-overlay {{
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0,0,0,0);
        cursor: pointer;
        border-radius: 8px;
    }}
    
    /* 🔥 新增：金手指单独一行样式 */
    .cheat-full-row {{
        width: 100% !important;
        margin-top: 16px !important;
        padding-top: 16px !important;
        border-top: 1px dashed #e0e0e0 !important;
    }}
    
    .cheat-value {{
        font-size: 13px !important;
        color: #333 !important;
        padding: 10px 12px !important;
        border-radius: 6px !important;
        background: #f0f4ff !important;
        border: 1px solid #d0d7ff !important;
        min-height: 40px !important;
        display: flex !important;
        align-items: center !important;
        cursor: pointer !important;
        transition: all 0.2s ease !important;
        word-break: break-word !important;
        line-height: 1.5 !important;
    }}
    
    .cheat-value:hover {{
        background: #e1e7ff !important;
        border-color: #a3b1ff !important;
    }}
    
    /* 🔥 修复：移除形状切换按钮的多余显示 */
    .shape-toggle-container {{
        display: none !important;
    }}
    
    /* 🔥 新增：隐藏按钮样式 */
    .hidden-button-container {{
        display: none !important;
    }}
    
    /* 🔥 新增：快速填充按钮样式 */
    .quick-fill-buttons {{
        display: flex !important;
        flex-wrap: wrap !important;
        gap: 5px !important;
        margin-top: 5px !important;
        margin-bottom: 10px !important;
    }}
    
    .quick-fill-btn {{
        padding: 4px 8px !important;
        border: 1px solid #ddd !important;
        border-radius: 4px !important;
        background: #f8f9fa !important;
        color: #666 !important;
        font-size: 11px !important;
        cursor: pointer !important;
        transition: all 0.2s ease !important;
    }}
    
    .quick-fill-btn:hover {{
        background: #e9ecef !important;
        border-color: #ced4da !important;
    }}
    
    .quick-fill-btn.active {{
        background: {THEME_LIGHT} !important;
        color: {THEME_COLOR} !important;
        border-color: {THEME_COLOR} !important;
    }}
    
    </style>
    """, unsafe_allow_html=True)
    
    all_books_res = db_mgr.query("SELECT id, title FROM books")
    all_books = {r['title']: int(r['id']) for r in all_books_res} if all_books_res else {}
    book_titles = list(all_books.keys())
    
    if not all_books:
        st.warning("数据库中没有书籍，请先在 [书籍管理] 中添加书籍。")
        return

    current_book_id = st.session_state.get('current_book_id')
    if not current_book_id:
        last_viewed = engine.get_config_db("last_viewed_book_id", None)
        if last_viewed and int(last_viewed) in all_books.values():
            current_book_id = int(last_viewed)
            st.session_state['current_book_id'] = current_book_id
    
    default_idx = 0
    if current_book_id:
        for idx, t in enumerate(book_titles):
            if all_books[t] == current_book_id:
                default_idx = idx
                break
    
    def on_book_change():
        new_title = st.session_state.character_manager_book_selector
        new_id = all_books[new_title]
        st.session_state['current_book_id'] = new_id
        engine.set_config_db("last_viewed_book_id", new_id)
        
    selected_title = st.selectbox(
        "📚 **选择要管理的角色书籍：**", 
        book_titles, 
        index=default_idx, 
        key="character_manager_book_selector",
        on_change=on_book_change
    )
    
    selected_book_id = int(all_books.get(selected_title))
    if selected_book_id != st.session_state.get('current_book_id'):
        st.session_state['current_book_id'] = selected_book_id
        engine.set_config_db("last_viewed_book_id", selected_book_id)
    
    current_book_id = selected_book_id
    current_book = None
    if current_book_id:
        res = db_mgr.query("SELECT * FROM books WHERE id=?", (current_book_id,))
        if res: current_book = res[0]
    if not current_book: return
    st.info(f"当前管理书籍：《{current_book['title']}》")
    
    col_graph, col_edit = st.columns([2, 1]) 

    # =================================================================
    # 左侧：人物关系图谱
    # =================================================================
    with col_graph:
        st.subheader("🕸️ 人物关系图谱")
        
        # 🔥 修复：简化形状切换按钮，移除多余显示
        if "graph_avatar_shape" not in st.session_state:
            st.session_state.graph_avatar_shape = "circle"
        
        # 使用简单的按钮式切换
        st.markdown('<div class="simple-shape-toggle">', unsafe_allow_html=True)
        col_shape1, col_shape2 = st.columns(2)
        
        with col_shape1:
            if st.button("⚪ 圆形", key="graph_shape_circle_btn", use_container_width=True):
                st.session_state.graph_avatar_shape = "circle"
                st.rerun()
        
        with col_shape2:
            if st.button("⬜ 方形", key="graph_shape_square_btn", use_container_width=True):
                st.session_state.graph_avatar_shape = "square"
                st.rerun()
        
        st.markdown('</div>', unsafe_allow_html=True)
        
        chars_graph_rows = db_mgr.query("SELECT * FROM characters WHERE book_id=?", (current_book_id,))
        
        if not chars_graph_rows:
            st.info("暂无角色，请在右侧添加。")
        else:
            chars_graph = [dict(r) for r in chars_graph_rows]
            saved_relations = load_relations_from_disk(current_book_id)
            has_cache = saved_relations is not None and len(saved_relations) > 0
            
            # 🔥 显示角色和关系数量
            relation_count = len(saved_relations) if saved_relations else 0
            st.caption(f"📊 角色: {len(chars_graph)} 人 | 关系: {relation_count} 条")
            
            # 🔥 修复：将生成图谱按钮放在形状切换下方
            btn_label = "🔄 重新生成图谱" if has_cache else "🤖 AI 生成图谱"
            if st.button(btn_label, key="gen_chart_btn", type="primary", use_container_width=True):
                assignments = engine.get_config_db("model_assignments", {})
                assigned_key = assignments.get("books_arch_gen")
                if not assigned_key: assigned_key = assignments.get("novel_structure_gen")
                if not assigned_key: assigned_key = "GPT_4o"

                # 🔥 修复：使用支持自定义模型的解析器
                client, model_name, model_key = _resolve_ai_client(engine, assigned_key)
                
                if not client: 
                    st.error(f"请先配置图谱生成模型")
                else:
                    # 🔥 审计：启动图谱生成
                    log_audit_event("AI辅助", "启动图谱生成", {"书籍": current_book['title']})
                    
                    with st.spinner("AI 正在阅读分析人物关系..."):
                        # 🔥 修复：使用本地的 ai_generate_relation_map 而不是 engine 的方法
                        ok, res = ai_generate_relation_map(engine, current_book_id, chars_graph, client, model_name)
                        
                        if ok and isinstance(res, list):
                            # 🔥 修复：合并新旧关系，而不是替换
                            current_rels = load_relations_from_disk(current_book_id) or []
                            # 构建现有关系指纹
                            existing_fingerprints = set()
                            for rel in current_rels:
                                fp = tuple(sorted([str(rel.get('source')), str(rel.get('target'))]))
                                existing_fingerprints.add(fp)
                            
                            # 添加新关系（避免重复）
                            new_rels_added = 0
                            for rel in res:
                                fp = tuple(sorted([str(rel.get('source')), str(rel.get('target'))]))
                                if fp not in existing_fingerprints:
                                    current_rels.append(rel)
                                    existing_fingerprints.add(fp)
                                    new_rels_added += 1
                            
                            save_relations_to_disk(current_book_id, current_rels)
                            saved_relations = current_rels
                            
                            # 🔥 审计：图谱生成完成
                            log_audit_event("AI辅助", "图谱生成完成", {"新增关系数": new_rels_added, "总关系数": len(current_rels)})
                            
                            st.success(f"✅ 图谱生成完成！新增 {new_rels_added} 条关系，共 {len(current_rels)} 条关系")
                            time.sleep(1.5)
                            st.rerun()
                        else:
                            st.error(f"生成失败: {res}")
            
            relations_data = saved_relations if saved_relations else []
            try:
                role_colors = {
                    "主角": "#d32f2f", "双主角": "#d32f2f", "反派BOSS": "#212121",
                    "主要配角": "#1976d2", "次要配角": "#64b5f6", "挚友/死党": "#388e3c",
                    "暗恋者/伴侣": "#e91e63", "导师/师父": "#fbc02d", "default": "#9e9e9e"
                }

                nodes = []
                node_ids = set()
                
                for char in chars_graph:
                    char_id = char['id']
                    node_ids.add(char_id)
                    
                    color = role_colors.get(char.get('role'), role_colors["default"])
                    # 🔥 修改：增加节点大小，主要角色80，次要角色60
                    size = 80 if char.get('is_major') else 60
                    
                    # 🔥 获取头像内容
                    avatar_path = char.get('avatar', '')
                    image_url = get_node_image_content(avatar_path)
                    
                    # 如果获取不到头像，使用默认头像
                    if not image_url:
                        image_url = get_default_avatar(char['name'])
                    
                    # 🔥 修复：根据选择的形状设置节点形状和边框圆角
                    if image_url:
                        if st.session_state.graph_avatar_shape == "circle":
                            shape = 'circularImage'
                        else:
                            shape = 'image'
                    else:
                        shape = 'dot'
                    
                    desc_raw = (char.get('desc') or "暂无描述").replace('\n', ' ').strip()
                    
                    # 🔥 优化：创建更完整的悬停信息
                    extra_info_parts = []
                    if char.get('power_level'): 
                        extra_info_parts.append(f"境界：{char.get('power_level')}")
                    if char.get('profession'): 
                        extra_info_parts.append(f"职业：{char.get('profession')}")
                    if char.get('origin'):
                        extra_info_parts.append(f"出身：{char.get('origin')}")
                    
                    extra_info = " | ".join(extra_info_parts) if extra_info_parts else ""
                    
                    # 🔥 修复悬停显示：使用更完整的HTML结构，确保自动换行
                    tooltip_html = f"""
                        <div style='max-width: 380px; word-wrap: break-word;'>
                            <strong>{html.escape(char['name'])}</strong>
                            <span class="meta">
                                定位：{char.get('role', '未知')} | 性别：{char.get('gender', '未知')} | 种族：{char.get('race', '未知')}
                            </span>
                            {f"<div class='info-line'>{html.escape(extra_info)}</div>" if extra_info else ""}
                            <div class="desc">{html.escape(desc_raw[:200])}{'...' if len(desc_raw) > 200 else ''}</div>
                        </div>
                    """
                    
                    nodes.append({
                        "id": char_id,
                        "label": char['name'][:6] + "..." if len(char['name']) > 6 else char['name'],
                        "title": tooltip_html, 
                        "shape": shape,
                        "image": image_url if image_url else None,
                        "size": size,
                        "borderWidth": 3, 
                        "borderWidthSelected": 5,
                        "color": { 
                            "border": color, 
                            "background": "#ffffff",
                            "highlight": {
                                "border": color,
                                "background": "#f0f0f0"
                            }
                        }
                    })
                    
                edges = []
                seen_edges = set()
                
                # 预先建立 id 到 name 的映射，方便在 tooltip 中显示名字
                id_name_map = {c['id']: c['name'] for c in chars_graph}
                
                for rel in relations_data:
                    source_id = rel.get('source')
                    target_id = rel.get('target')
                    
                    if source_id in node_ids and target_id in node_ids:
                        edge_key = tuple(sorted([source_id, target_id]))
                        
                        if edge_key in seen_edges:
                            continue
                        seen_edges.add(edge_key)
                        
                        full_label = rel.get('label', '')
                        reason = rel.get('reason', '')
                        
                        # 🔥 优化1：使用智能缩短函数 - 简略显示
                        display_label = get_smart_short_label(full_label)
                            
                        # 🔥 优化2：鼠标悬停显示简单文本
                        src_name = id_name_map.get(source_id, '?')
                        tgt_name = id_name_map.get(target_id, '?')
                        
                        # 构建简单文本悬停提示
                        tooltip_text = f"{src_name} ↔ {tgt_name}\n关系: {full_label}"
                        if reason:
                            tooltip_text += f"\n原因: {reason}"
                        
                        edges.append({
                            "from": source_id,
                            "to": target_id,
                            "label": display_label, 
                            "title": tooltip_text,    
                            "width": 3,  # 🔥 增加线宽
                            # 🔥 优化3：美化连线字体背景
                            "font": {
                                "size": 11,
                                "color": "#333333",
                                "align": "middle",
                                "background": "rgba(255,255,255,0.85)", 
                                "strokeWidth": 0
                            },
                            "color": {"color": "#444444", "highlight": "#2e7d32"}  # 🔥 加深连线颜色
                        })
                
                if nodes:
                    html_code = generate_graph_html(nodes, edges, height="450px")
                    components.html(html_code, height=470, scrolling=False)
                else:
                    st.info("没有可显示的节点")
                
            except Exception as e:
                st.error(f"渲染构建错误: {str(e)}")
                import traceback
                st.code(traceback.format_exc())

    # =================================================================
    # 右侧：编辑与列表
    # =================================================================
    with col_edit:
        tab_add, tab_rels, tab_list, tab_ai = st.tabs(["➕ 添加角色", "🔗 关系管理", "📋 列表管理", "🤖 AI 提取"])
        
        # --- Tab 1: 手动添加 ---
        with tab_add:
            st.caption("添加新角色并可直接绑定关系。")
            name = st.text_input("姓名", key="manual_name")
            
            # 🔥 修改：将下拉选择改为文本输入，并添加快捷填充按钮
            kp_man = "man" 
            
            # 定位输入
            st.markdown('<div class="field-label">定位</div>', unsafe_allow_html=True)
            role_sel = st.text_input("", key=f"{kp_man}_role", placeholder="例如：主角、反派...", label_visibility="collapsed")
            # 定位快捷填充按钮
            st.markdown('<div class="quick-fill-buttons">', unsafe_allow_html=True)
            role_options = st.session_state.role_options
            col_roles = st.columns(4)
            for idx, role in enumerate(role_options[:12]):  # 只显示前12个选项
                with col_roles[idx % 4]:
                    if st.button(role, key=f"quick_role_{role}_{kp_man}", use_container_width=True):
                        st.session_state[f"{kp_man}_role"] = role
                        st.rerun()
            st.markdown('</div>', unsafe_allow_html=True)
            
            # 性别输入
            st.markdown('<div class="field-label">性别</div>', unsafe_allow_html=True)
            gen_sel = st.text_input("", key=f"{kp_man}_gen", placeholder="例如：男、女...", label_visibility="collapsed")
            # 性别快捷填充按钮
            st.markdown('<div class="quick-fill-buttons">', unsafe_allow_html=True)
            gender_options = st.session_state.gender_options
            col_genders = st.columns(3)
            for idx, gender in enumerate(gender_options):
                with col_genders[idx % 3]:
                    if st.button(gender, key=f"quick_gender_{gender}_{kp_man}", use_container_width=True):
                        st.session_state[f"{kp_man}_gen"] = gender
                        st.rerun()
            st.markdown('</div>', unsafe_allow_html=True)
            
            # 种族输入
            st.markdown('<div class="field-label">种族</div>', unsafe_allow_html=True)
            race_sel = st.text_input("", key=f"{kp_man}_race", placeholder="例如：人族、精灵...", label_visibility="collapsed")
            # 种族快捷填充按钮
            st.markdown('<div class="quick-fill-buttons">', unsafe_allow_html=True)
            race_options = st.session_state.race_options
            col_races = st.columns(4)
            for idx, race in enumerate(race_options[:12]):  # 只显示前12个选项
                with col_races[idx % 4]:
                    if st.button(race, key=f"quick_race_{race}_{kp_man}", use_container_width=True):
                        st.session_state[f"{kp_man}_race"] = race
                        st.rerun()
            st.markdown('</div>', unsafe_allow_html=True)
            
            up_new_add = st.file_uploader("上传头像", type=['jpg','png'], key="man_up", label_visibility="collapsed")
            av_url = st.text_input("或 头像 URL", key="manual_av")
            
            with st.expander("📝 详细设定 (身份/能力/关系...)", expanded=False):
                st.caption("以下内容将存入数据库，供 AI 写作参考")
                c_d1, c_d2 = st.columns(2)
                m_origin = c_d1.text_input("出身背景", placeholder="贵族/平民/穿越...", key="m_origin")
                m_prof = c_d2.text_input("职业/天赋", placeholder="剑士/炼药师...", key="m_prof")
                
                m_cheat = st.text_input("金手指/核心能力", placeholder="系统/神秘宝物", key="m_cheat")
                c_d3, c_d4 = st.columns(2)
                m_level = c_d3.text_input("当前境界", placeholder="斗之气三段", key="m_level")
                m_limit = c_d4.text_input("能力代价", placeholder="消耗寿命/冷却...", key="m_limit")
                
                m_face = st.text_input("外貌特征", placeholder="一道疤/异色瞳", key="m_face")
                m_sign = st.text_input("标志性物品/动作", placeholder="摸鼻子/如意棒", key="m_sign")
                
                m_rel_pro = st.text_input("与主角关系", placeholder="盟友/死敌", key="m_rel_pro")
                m_social = st.text_input("社会角色/恩仇", placeholder="家族族长/欠某人情", key="m_social")
            
            desc = st.text_area("综合简介", height=150, key="manual_desc")
            
            st.markdown("---")
            st.markdown("**🔗 初始关系绑定**")
            
            char_rows = db_mgr.query("SELECT id, name FROM characters WHERE book_id=?", (current_book_id,))
            char_options = {c['name']: c['id'] for c in char_rows}
            char_names = ["(无)"] + list(char_options.keys())
            
            rel_target_name = st.selectbox("关联对象", char_names, key="man_rel_target")
            rel_desc = st.text_input("关系描述 (如: 义妹)", key="man_rel_desc")

            if st.button("确认添加", type="primary", width="stretch"):
                if name:
                    # 🔥 手动添加的去重逻辑
                    exist_check = db_mgr.query("SELECT id FROM characters WHERE book_id=? AND name=?", (current_book_id, name))
                    if exist_check:
                        st.warning(f"角色 {name} 已存在，请勿重复添加。")
                    else:
                        final_av = av_url
                        if up_new_add:
                             temp_id = int(time.time())
                             saved_path = save_avatar_file(up_new_add, temp_id)
                             if saved_path: 
                                 final_av = saved_path
                        
                        if not final_av:
                             final_av = generate_bing_search_image(f"{current_book['title']} {name} 插画")
                             if not final_av:
                                final_av = get_default_avatar(name)

                        new_char_id = db_mgr.execute(
                            """INSERT INTO characters (
                                book_id, name, role, gender, race, desc, is_major, avatar,
                                origin, profession, cheat_ability, power_level, ability_limitations,
                                appearance_features, signature_sign, relationship_to_protagonist, social_role
                            ) VALUES (?,?,?,?,?,?,?,?, ?,?,?,?,?, ?,?,?,?)""",
                            (
                                current_book_id, name, role_sel, gen_sel, race_sel, desc, True, final_av,
                                m_origin, m_prof, m_cheat, m_level, m_limit,
                                m_face, m_sign, m_rel_pro, m_social
                            )
                        )
                        
                        if rel_target_name != "(无)" and rel_desc:
                            target_id = char_options[rel_target_name]
                            current_relations = load_relations_from_disk(current_book_id) or []
                            current_relations.append({
                                "source": new_char_id,
                                "target": target_id,
                                "label": rel_desc,
                                "weight": 1
                            })
                            save_relations_to_disk(current_book_id, current_relations)

                        update_book_timestamp_by_book_id(current_book_id)
                        
                        # 🔥 审计：手动添加
                        log_audit_event("角色管理", "手动添加角色", {"角色名": name, "定位": role_sel, "书ID": current_book_id})
                        
                        st.toast(f"✅ {name} 已添加！")
                        time.sleep(0.5)
                        st.rerun()

        # --- Tab 2: 关系管理 ---
        with tab_rels:
            st.caption("管理角色间的连线。")
            char_rows = db_mgr.query("SELECT id, name FROM characters WHERE book_id=?", (current_book_id,))
            char_options = {c['name']: c['id'] for c in char_rows}
            char_names = list(char_options.keys())
            
            if not char_names:
                st.info("请先添加至少两个角色")
            else:
                c_src, c_tgt = st.columns([1, 1])
                with c_src:
                    src_name = st.selectbox("角色 A", char_names, key="rel_src")
                with c_tgt:
                    tgt_name = st.selectbox("角色 B", char_names, key="rel_tgt")
                    
                if src_name and tgt_name and src_name != tgt_name:
                    src_id = char_options[src_name]
                    tgt_id = char_options[tgt_name]
                    
                    current_relations = load_relations_from_disk(current_book_id) or []
                    existing_rel = None
                    existing_idx = -1
                    
                    for i, r in enumerate(current_relations):
                        if (r['source'] == src_id and r['target'] == tgt_id) or (r['source'] == tgt_id and r['target'] == src_id):
                            existing_rel = r
                            existing_idx = i
                            break
                    
                    rel_label = st.text_input("关系描述", value=existing_rel['label'] if existing_rel else "", key="rel_label_input")
                    
                    c_act_1, c_act_2 = st.columns([1, 1])
                    
                    if existing_rel:
                        with c_act_1:
                            if st.button("更新关系", type="primary", width="stretch"):
                                current_relations[existing_idx]['label'] = rel_label
                                save_relations_to_disk(current_book_id, current_relations)
                                
                                # 🔥 审计：更新关系
                                log_audit_event("关系管理", "更新关系", {"源角色": src_name, "目标角色": tgt_name, "新描述": rel_label})
                                
                                st.toast("✅ 关系已更新")
                                time.sleep(0.5)
                                st.rerun()
                        with c_act_2:
                            if st.button("删除连线", type="secondary", width="stretch"):
                                current_relations.pop(existing_idx)
                                save_relations_to_disk(current_book_id, current_relations)
                                
                                # 🔥 审计：删除关系
                                log_audit_event("关系管理", "删除关系", {"源角色": src_name, "目标角色": tgt_name}, status="WARNING")
                                
                                st.toast("🗑️ 关系已删除")
                                time.sleep(0.5)
                                st.rerun()
                    else:
                        if st.button("➕ 建立新关系", type="primary", width="stretch"):
                            new_rel = {
                                "source": src_id,
                                "target": tgt_id,
                                "label": rel_label,
                                "weight": 1
                            }
                            current_relations.append(new_rel)
                            save_relations_to_disk(current_book_id, current_relations)
                            
                            # 🔥 审计：新建关系
                            log_audit_event("关系管理", "新建关系", {"源角色": src_name, "目标角色": tgt_name, "描述": rel_label})
                            
                            st.toast("✅ 关系已建立")
                            time.sleep(0.5)
                            st.rerun()

        # --- Tab 3: 列表管理 (完全重写折叠栏布局) ---
        with tab_list:
            c_count, c_view = st.columns([2, 1])
            with c_count:
                 rows = db_mgr.query("SELECT * FROM characters WHERE book_id=? ORDER BY is_major DESC, id DESC", (current_book_id,))
                 count = len(rows) if rows else 0
                 st.markdown(f"**共 {count} 名角色** <span style='color:grey;font-size:0.8em'>(已按重要性排序)</span>", unsafe_allow_html=True)
            
            # 🔥 修复：简化列表管理的头像形状切换
            if "list_avatar_shape" not in st.session_state:
                st.session_state.list_avatar_shape = "circle"
            
            with c_view:
                # 使用简单的文本标签
                st.markdown("**头像形状:**", unsafe_allow_html=True)
                col_shape1, col_shape2 = st.columns(2)
                with col_shape1:
                    if st.button("⚪", key="list_shape_circle_btn", use_container_width=True):
                        st.session_state.list_avatar_shape = "circle"
                        st.rerun()
                with col_shape2:
                    if st.button("⬜", key="list_shape_square_btn", use_container_width=True):
                        st.session_state.list_avatar_shape = "square"
                        st.rerun()
            
            radius_style = "50%" if st.session_state.list_avatar_shape == "circle" else "8px"
            
            if not rows:
                st.info("暂无角色")
            else:
                all_chars_res = [dict(r) for r in rows]
                all_chars_res.sort(key=lambda x: get_role_priority(x.get('role')))

                for ch in all_chars_res:
                    expander_title = f"{ch['name']} ({ch.get('role', '未知')})"
                    
                    with st.expander(expander_title, expanded=False):
                        # 🔥 完全重写：使用Streamlit原生列布局，避免CSS问题
                        col1, col2 = st.columns([1, 1], gap="large")
                        
                        with col1:
                            # 左侧列：头像、名字、性别、种族
                            
                            # 🔥 修改：头像可点击编辑，移除编辑按钮
                            av_src = get_node_image_content(ch.get('avatar')) or get_default_avatar(ch['name'])
                            
                            # 使用可点击的容器包装头像
                            clickable_avatar_html = f'''
                            <div class="clickable-avatar-container">
                                <img src="{av_src}" class="clickable-avatar-img" style="border-radius: {radius_style};"
                                     onclick="document.getElementById('avatar_click_{ch['id']}').click()">
                                <div class="avatar-overlay" onclick="document.getElementById('avatar_click_{ch['id']}').click()"></div>
                            </div>
                            '''
                            st.markdown(clickable_avatar_html, unsafe_allow_html=True)
                            
                            # 隐藏的按钮容器，用于触发编辑
                            with st.container():
                                st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                                if st.button("编辑头像", key=f"avatar_click_{ch['id']}", type="secondary", use_container_width=True):
                                    edit_avatar_dialog(ch['id'], ch.get('avatar'), ch['name'], current_book_id)
                                st.markdown('</div>', unsafe_allow_html=True)
                            
                            # 名字 - 可点击编辑
                            name_value = ch.get("name", "未设置") or ""
                            st.markdown(f'''
                            <div class="field-row" onclick="document.getElementById('edit_name_{ch["id"]}').click()">
                                <div class="field-label">名字</div>
                                <div class="field-value">{name_value}</div>
                            </div>
                            ''', unsafe_allow_html=True)
                            
                            # 隐藏的编辑按钮
                            with st.container():
                                st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                                if st.button("编辑名字", key=f"edit_name_{ch['id']}", type="secondary", use_container_width=True):
                                    edit_field_dialog(ch['id'], "姓名", ch.get('name'), ch['name'], current_book_id)
                                st.markdown('</div>', unsafe_allow_html=True)
                            
                            # 性别 - 可点击编辑
                            gender_value = ch.get("gender", "未设置") or ""
                            st.markdown(f'''
                            <div class="field-row" onclick="document.getElementById('edit_gender_{ch["id"]}').click()">
                                <div class="field-label">性别</div>
                                <div class="field-value">{gender_value}</div>
                            </div>
                            ''', unsafe_allow_html=True)
                            
                            # 隐藏的编辑按钮
                            with st.container():
                                st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                                if st.button("编辑性别", key=f"edit_gender_{ch['id']}", type="secondary", use_container_width=True):
                                    edit_field_dialog(ch['id'], "性别", ch.get('gender'), ch['name'], current_book_id)
                                st.markdown('</div>', unsafe_allow_html=True)
                            
                            # 种族 - 可点击编辑
                            race_value = ch.get("race", "未设置") or ""
                            st.markdown(f'''
                            <div class="field-row" onclick="document.getElementById('edit_race_{ch["id"]}').click()">
                                <div class="field-label">种族</div>
                                <div class="field-value">{race_value}</div>
                            </div>
                            ''', unsafe_allow_html=True)
                            
                            # 隐藏的编辑按钮
                            with st.container():
                                st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                                if st.button("编辑种族", key=f"edit_race_{ch['id']}", type="secondary", use_container_width=True):
                                    edit_field_dialog(ch['id'], "种族", ch.get('race'), ch['name'], current_book_id)
                                st.markdown('</div>', unsafe_allow_html=True)
                        
                        with col2:
                            # 右侧列：定位、出身、境界、职业/天赋
                            
                            # 定位 - 可点击编辑
                            role_value = ch.get("role", "未设置") or ""
                            st.markdown(f'''
                            <div class="field-row" onclick="document.getElementById('edit_role_{ch["id"]}').click()">
                                <div class="field-label">定位</div>
                                <div class="field-value">{role_value}</div>
                            </div>
                            ''', unsafe_allow_html=True)
                            
                            # 隐藏的编辑按钮
                            with st.container():
                                st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                                if st.button("编辑定位", key=f"edit_role_{ch['id']}", type="secondary", use_container_width=True):
                                    edit_field_dialog(ch['id'], "定位", ch.get('role'), ch['name'], current_book_id)
                                st.markdown('</div>', unsafe_allow_html=True)
                            
                            # 出身 - 可点击编辑
                            origin_value = ch.get("origin", "未设置") or ""
                            st.markdown(f'''
                            <div class="field-row" onclick="document.getElementById('edit_origin_{ch["id"]}').click()">
                                <div class="field-label">出身</div>
                                <div class="field-value">{origin_value}</div>
                            </div>
                            ''', unsafe_allow_html=True)
                            
                            # 隐藏的编辑按钮
                            with st.container():
                                st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                                if st.button("编辑出身", key=f"edit_origin_{ch['id']}", type="secondary", use_container_width=True):
                                    edit_field_dialog(ch['id'], "出身背景", ch.get('origin'), ch['name'], current_book_id)
                                st.markdown('</div>', unsafe_allow_html=True)
                            
                            # 境界 - 可点击编辑
                            power_level_value = ch.get("power_level", "未设置") or ""
                            st.markdown(f'''
                            <div class="field-row" onclick="document.getElementById('edit_power_level_{ch["id"]}').click()">
                                <div class="field-label">境界</div>
                                <div class="field-value">{power_level_value}</div>
                            </div>
                            ''', unsafe_allow_html=True)
                            
                            # 隐藏的编辑按钮
                            with st.container():
                                st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                                if st.button("编辑境界", key=f"edit_power_level_{ch['id']}", type="secondary", use_container_width=True):
                                    edit_field_dialog(ch['id'], "当前境界/等级", ch.get('power_level'), ch['name'], current_book_id)
                                st.markdown('</div>', unsafe_allow_html=True)
                            
                            # 职业/天赋 - 可点击编辑
                            profession_value = ch.get("profession", "未设置") or ""
                            st.markdown(f'''
                            <div class="field-row" onclick="document.getElementById('edit_profession_{ch["id"]}').click()">
                                <div class="field-label">职业/天赋</div>
                                <div class="field-value">{profession_value}</div>
                            </div>
                            ''', unsafe_allow_html=True)
                            
                            # 隐藏的编辑按钮
                            with st.container():
                                st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                                if st.button("编辑职业", key=f"edit_profession_{ch['id']}", type="secondary", use_container_width=True):
                                    edit_field_dialog(ch['id'], "职业/天赋", ch.get('profession'), ch['name'], current_book_id)
                                st.markdown('</div>', unsafe_allow_html=True)
                        
                        # 🔥 金手指单独一行 - 可点击编辑
                        cheat_value = ch.get("cheat_ability") or "未设置"
                        # 🔥 修改：如果金手指为"None"，显示为空
                        if cheat_value == "None" or cheat_value == "none" or cheat_value == "null" or cheat_value == "None":
                            cheat_value = "未设置"
                            
                        st.markdown(f'''
                        <div class="field-row" onclick="document.getElementById('edit_cheat_{ch["id"]}').click()">
                            <div class="field-label">金手指</div>
                            <div class="cheat-value">{cheat_value}</div>
                        </div>
                        ''', unsafe_allow_html=True)
                        
                        # 隐藏的编辑按钮
                        with st.container():
                            st.markdown('<div class="hidden-button-container">', unsafe_allow_html=True)
                            if st.button("编辑金手指", key=f"edit_cheat_{ch['id']}", type="secondary", use_container_width=True):
                                edit_field_dialog(ch['id'], "金手指/核心能力", ch.get('cheat_ability'), ch['name'], current_book_id)
                            st.markdown('</div>', unsafe_allow_html=True)
                        
                        # --- 🔥 人物小传 ---
                        st.markdown('<div style="margin-top: 20px; padding-top: 15px; border-top: 1px solid #e0e0e0;">', unsafe_allow_html=True)
                        st.markdown('<div class="field-label">人物小传</div>', unsafe_allow_html=True)
                        st.markdown('<div class="field-hint">角色的背景故事、性格特点、重要经历等，供AI深度理解角色</div>', unsafe_allow_html=True)
                        
                        # 人物简介
                        n_desc = st.text_area("", ch.get('desc',''), height=120, key=f"d_{ch['id']}", 
                                             label_visibility="collapsed", 
                                             placeholder="例：戴沐白，星罗帝国皇子，邪眸白虎武魂。性格狂野霸气，好色但有担当，作为史莱克七怪的老大，拥有极强的领导力和责任感...")
                        
                        # 🔥 底部按钮栏
                        st.markdown('<div style="display: flex; gap: 8px; margin-top: 15px;">', unsafe_allow_html=True)
                        
                        # 🔥 删除按钮
                        if st.button("🗑️ 删除", key=f"del_{ch['id']}", type="secondary", use_container_width=True):
                            db_mgr.execute("DELETE FROM characters WHERE id=?", (ch['id'],))
                            update_book_timestamp_by_book_id(current_book_id)
                            
                            # 🔥 审计：删除角色
                            log_audit_event("角色管理", "删除角色", {"ID": ch['id'], "姓名": ch['name']}, status="WARNING")
                            st.rerun()
                        
                        # 🔥 保存按钮
                        if st.button("💾 保存小传", key=f"sav_{ch['id']}", type="primary", use_container_width=True):
                            db_mgr.execute("""
                                UPDATE characters SET desc=? WHERE id=?
                            """, (n_desc, ch['id']))
                            update_book_timestamp_by_book_id(current_book_id)
                            
                            # 🔥 审计：更新角色信息
                            log_audit_event("角色管理", "更新角色描述", {"ID": ch['id'], "姓名": ch['name']})
                            
                            st.toast("✅ 保存成功")
                            st.rerun()
                        
                        st.markdown('</div>', unsafe_allow_html=True)
                        st.markdown('</div>', unsafe_allow_html=True)

        # --- Tab 4: AI 提取 ---
        with tab_ai:
            st.caption("AI 将智能分析小说内容，自动提取角色并建立关系网。")
            
            # 🔥 修复：确保AI提取功能有按钮和进度条
            feature_key = "character_extract"
            assigned_model_key = engine.get_config_db("model_assignments", {}).get(feature_key, FEATURE_MODELS[feature_key]['default'])
            
            # 🔥 修复：使用支持自定义模型的解析器
            client, model_name, _ = _resolve_ai_client(engine, assigned_model_key)
            
            if not client:
                st.error(f"❌ AI 模型未配置。请在【系统设置】->【功能调度】中为 [角色提取] 选择模型并保存。")
            else:
                # 显示当前配置的模型
                display_model_name = MODEL_MAPPING.get(assigned_model_key, {}).get('name', model_name)
                st.info(f"当前使用模型：**{display_model_name}**")
                
                if st.button("🚀 启动智能分析与提取", type="primary", width="stretch"):
                    progress_bar = st.progress(0)
                    status_text = st.empty()
                    
                    def update_progress(p, text):
                        progress_bar.progress(p)
                        status_text.text(text)
                    
                    ok, result = ai_extract_characters(engine, db_mgr, current_book, current_book_id, update_progress)
                    
                    if ok:
                        st.session_state[f"extracted_chars_{current_book_id}"] = result
                        st.success(f"分析完成！共发现 {len(result)} 个新角色")
                        time.sleep(1) 
                        st.rerun()
                    else:
                        st.error(result)

            extracted_data = st.session_state.get(f"extracted_chars_{current_book_id}", [])
            
            if extracted_data:
                st.divider()
                st.write(f"📊 **待确认导入角色：{len(extracted_data)} 人**")
                
                if st.button("📥 全部导入 (含角色关系)", type="primary", width="stretch"):
                    
                    # 🔥 1. 新增：辅助函数，防止字典插入报错
                    def _safe_str(val):
                        if isinstance(val, (dict, list)):
                            try: return json.dumps(val, ensure_ascii=False)
                            except: return str(val)
                        return str(val) if val else ""

                    # 🔥 2. 批量插入角色 - 核心去重逻辑
                    name_id_map = {} 
                    
                    # 获取库里已有角色
                    existing_rows = db_mgr.query("SELECT id, name FROM characters WHERE book_id=?", (current_book_id,))
                    # 使用 strip() 防止空格导致的误判
                    for r in existing_rows: 
                        name_id_map[r['name'].strip()] = r['id']

                    pending_relations = []
                    new_char_count = 0

                    for char_data in extracted_data:
                        raw_name = char_data.get('name', '').strip()
                        if not raw_name: 
                            continue
                        
                        # 🔥 如果已存在，则记录 ID 并跳过插入
                        if raw_name in name_id_map:
                            print(f"角色 [{raw_name}] 已存在，跳过创建。")
                            # 仍需处理该角色的关系数据，所以不做 continue
                        else:
                            try:
                                cid = db_mgr.execute(
                                    """INSERT INTO characters (
                                        book_id, name, role, gender, race, desc, is_major, avatar,
                                        origin, profession, cheat_ability, power_level, ability_limitations,
                                        appearance_features, signature_sign, relationship_to_protagonist, social_role, debts_and_feuds
                                    ) VALUES (?,?,?,?,?,?,?,?, ?,?,?,?,?, ?,?,?,?,?)""",
                                    (
                                        current_book_id, 
                                        raw_name, 
                                        char_data.get('role', '路人'), 
                                        char_data.get('gender', '未知'), 
                                        char_data.get('race', '人族'), 
                                        char_data.get('desc', ''), 
                                        char_data.get('is_major', False), 
                                        char_data.get('avatar', ''),
                                        char_data.get('origin', ''),
                                        char_data.get('profession', ''),
                                        
                                        # 🔥 核心修复：使用 _safe_str 包裹可能出错的复杂字段
                                        _safe_str(char_data.get('cheat_ability', '')),
                                        _safe_str(char_data.get('power_level', '')),
                                        _safe_str(char_data.get('ability_limitations', '')),
                                        
                                        char_data.get('appearance_features', ''),
                                        char_data.get('signature_sign', ''),
                                        char_data.get('relationship_to_protagonist', ''),
                                        char_data.get('social_role', ''),
                                        char_data.get('debts_and_feuds', '')
                                    )
                                )
                                name_id_map[raw_name] = cid
                                new_char_count += 1
                            except Exception as e: 
                                print(f"Insert Error: {e}")
                                pass
                        
                        # 收集该角色的关系数据
                        if 'relationships' in char_data and isinstance(char_data['relationships'], list):
                            for rel in char_data['relationships']:
                                pending_relations.append({
                                    "source_name": raw_name,
                                    "target_name": rel.get('target', '').strip(),
                                    "label": rel.get('label', '相关')
                                })
                    
                    # 🔥 3. 处理关系导入
                    current_rels = load_relations_from_disk(current_book_id) or []
                    new_rel_count = 0
                    
                    # 构建现有关系的指纹，防止重复添加
                    existing_rel_fingerprints = set()
                    for r in current_rels:
                        # 关系是无向的，排序后作为指纹
                        fp = tuple(sorted([str(r['source']), str(r['target'])]))
                        existing_rel_fingerprints.add(fp)

                    for pr in pending_relations:
                        s_id = name_id_map.get(pr['source_name'])
                        t_id = name_id_map.get(pr['target_name'])
                        
                        if s_id and t_id and s_id != t_id:
                            fp = tuple(sorted([str(s_id), str(t_id)]))
                            
                            if fp not in existing_rel_fingerprints:
                                current_rels.append({
                                    "source": s_id,
                                    "target": t_id,
                                    "label": pr['label'],
                                    "weight": 1
                                })
                                existing_rel_fingerprints.add(fp) # 立即加入指纹
                                new_rel_count += 1
                    
                    save_relations_to_disk(current_book_id, current_rels)

                    st.session_state[f"extracted_chars_{current_book_id}"] = []
                    update_book_timestamp_by_book_id(current_book_id)
                    
                    # 🔥 审计：AI批量导入
                    log_audit_event("角色管理", "AI批量导入", {"新增角色数": new_char_count, "新增关系数": new_rel_count})
                    
                    st.toast(f"导入成功！新增角色 {new_char_count} 人，建立关系 {new_rel_count} 条。")
                    time.sleep(1)
                    st.rerun()
                    
                # 预览列表
                for idx, char_data in enumerate(extracted_data):
                    with st.expander(f"{idx+1}. {char_data['name']} ({char_data.get('role')})"):
                        c_p1, c_p2 = st.columns([1, 4], vertical_alignment="center")
                        with c_p1:
                            if char_data.get('avatar'):
                                st.image(char_data['avatar'], width=60)
                        with c_p2:
                            # 🔥 修复：增加性别和种族的显示
                            st.write(f"**[{char_data.get('gender', '未知')} | {char_data.get('race', '未知')}]** {char_data.get('desc')}")
                            if char_data.get('relationships'):
                                rels = [f"{r['target']}({r['label']})" for r in char_data['relationships']]
                                st.caption(f"🔗 关系: {', '.join(rels)}")
                            if char_data.get('cheat_ability'):
                                # 🔥 修改：金手指不显示"None"
                                cheat_value = char_data.get('cheat_ability')
                                if cheat_value and cheat_value != "None" and cheat_value != "none":
                                    st.caption(f"✨ 能力: {cheat_value}")
            else:
                st.info("点击上方按钮启动AI智能分析，提取小说中的角色。")